/***********************************************************************
 @C-file{
    author              = "Nelson H. F. Beebe",
    version             = "1.01",
    date                = "09 May 1998",
    time                = "22:56:42 MDT",
    filename            = "isbn.c",
    address             = "Center for Scientific Computing
			   University of Utah
			   Department of Mathematics, 322 INSCC
			   155 S 1400 E RM 233
			   Salt Lake City, UT 84112-0090
			   USA",
    telephone           = "+1 801 581 5254",
    FAX                 = "+1 801 585 1640, +1 801 581 4148",
    URL                 = "http://www.math.utah.edu/~beebe",
    checksum            = "54390 923 3137 25116",
    email               = "beebe@math.utah.edu, beebe@acm.org,
                           beebe@ieee.org (Internet)",
    codetable           = "ISO/ASCII",
    keywords            = "bibliography, ISBN, hyphenation",
    supported           = "yes",
    docstring           = "This file contains code for hyphenating
			   International Standard Book Numbers (ISBNs),
			   using the function ISBN_hyphenate(s,t,maxs).
			   No other public objects are defined by this
			   file.

			   If this file is compiled with the
			   preprocessor symbol TEST defined, then a
			   standalone program is produced that can be
			   used to filter test data containing ISBN
			   key/value pairs extracted from BibTeX files.
			   For example, the UNIX commands

			   bibclean -no-warn -max-width 0 *.bib | \
				grep '^ *ISBN *=' >tmpfile
			   sed -e 's/-//g' tmpfile | ./isbn | diff tmpfile -

			   should display no differences in ISBN
			   numbers, except where their hyphenation was
			   originally incorrect, or missing.

			   The checksum field above contains a CRC-16
			   checksum as the first value, followed by the
			   equivalent of the standard UNIX wc (word
			   count) utility output of lines, words, and
			   characters.  This is produced by Robert
			   Solovay's checksum utility.",
 }
***********************************************************************/

#include <config.h>
#include "xstdlib.h"
#include "xstring.h"
#include "xctype.h"

RCSID("$Id: isbn.c,v 1.4 1999/05/20 13:27:53 beebe Exp beebe $")

#define isISBNdigit(c)	(isdigit(c) || ((c) == 'X') || ((c) == 'x'))

#define MAX_ISBN	14 /* array size for complete ISBN and terminal NUL */

void			ISBN_hyphenate ARGS((char *s_,char *t_,size_t maxs_));

static const char	*fix_ISBN ARGS((const char *ISBN_));
static const char	*hyphenate_one_ISBN ARGS((const char *prefix_,
						  const char *ISBN_));
static int		in_ISBN_range ARGS((const char *begin_,
					    const char *ISBN_,
					    const char *end_));
static const char 	*next_ISBN ARGS((const char *s_, const char **end_));
static void		squeeze_ISBN ARGS((char * out_ISBN_,
					   const char *in_ISBN_));

#if defined(TEST)

#define MAX_BUF		4096

int			main ARGS((int argc_, char* argv_[]));

#if defined(HAVE_STDC)
int
main(int argc, char* argv[])
#else
int
main(argc,argv)
int argc;
char* argv[];
#endif
{
    char buf[MAX_BUF];
    char buf2[MAX_BUF];

    while (fgets(buf,MAX_BUF,stdin) != (char*)NULL)
    {
	ISBN_hyphenate(buf,buf2,MAX_BUF);
	(void)fputs(buf,stdout);
	(void)fflush(stdout);
    }

    exit (EXIT_SUCCESS);
    return (EXIT_SUCCESS);
}
#endif /* defined(TEST) */

#if defined(HAVE_STDC)
static const char *
next_ISBN(const char *s,const char **next)
#else /* K&R style */
static const char *
next_ISBN(s,next)
const char *s;
const char **next;
#endif
{
    size_t n;
    const char *start;

    while (*s)				/* scan over s[] */
    {
	for ( ; *s && !isISBNdigit(*s); ++s) /* ignore non-ISBN digits */
	    /* NO-OP */;

	for (n = 0, start = s; *s; ++s)	/* scan over ISBN */
	{
	    if (isISBNdigit(*s))
	    {
		n++;
		if (n == 10)		/* then we found an ISBN */
		{
		    *next = s + 1;
		    return (start);
		}
	    }
	    else if ((*s == ' ') || (*s == '-'))
		/* NO-OP */;
	    else
		break;
	}
    }
    *next = (const char*)NULL;
    return ((const char*)NULL);		/* no ISBN recognized */
}

    /*******************************************************************
    NB: We intentionally include the hyphen that separates the
    countrygroupnumber from the publishernumber, in order to improve
    readability, even though this complicates the processing in
    ISBN-in-rangep.  Given the position of that hyphen, we automatically
    compute the required hyphen positions.
    *******************************************************************/

#if defined(HAVE_STDC)
static const char *
fix_ISBN(const char *ISBN)
#else /* K&R style */
static const char *
fix_ISBN(ISBN)
const char *ISBN;
#endif
{
    size_t i;

    static struct
    {
	const char *begin;
	const char *end;
    } ISBN_range[] =
    {
     /* Australia, Canada (English), Gibraltar, Ireland, New Zealand,
	South Africa, Swaziland, United Kingdom, United States of America,
	Zimbabwe */
     { "0-00",			"0-19" },
     { "0-200",			"0-698" },
     { "0-7000",		"0-8499" },
     { "0-85000",		"0-89997" },
     { "0-900000",		"0-949999" },
     { "0-9500000",		"0-9801223" },

     /* Australia, Canada (English), Gibraltar, Ireland, New Zealand,
	South Africa, Swaziland, United Kingdom, United States of America,
	Zimbabwe */
     { "1-55000",		"1-86957" },
     { "1-869800",		"1-895682" },

     /* Belgium, Canada (French), France, Luxembourg, Switzerland (French) */
     { "2-01",			"2-19" },
     { "2-200",			"2-607" },
     { "2-7000",		"2-8317" },
     { "2-84001",		"2-89436" },
     { "2-900000",		"2-940028" },
     { "2-9500000",		"2-9900003" },

     /* Austria, Germany, Switzerland (German) */
     { "3-01",			"3-19" },
     { "3-201",			"3-633" },
     { "3-7001",		"3-8252" },
     { "3-85001",		"3-89524" },
     { "3-900001",		"3-929031" },
     { "3-9500000",		"3-9802964" },

     /* Japan */
     { "4-00",			"4-19" },
     { "4-250",			"4-657" },
     { "4-7500",		"4-8470" },
     { "4-87000",		"4-89829" },
     { "4-900000",		"4-949999" },

     /* Azerbaijan, Belarus, Commonwealth of Independent States, Estonia,
	Georgia, Kazakhstan, Kyrgyzstan, Latvia, Lithuania, Tajikistan,
	Turkmenistan, Uzbekistan */

     { "5-01",			"5-12" },
     { "5-200",			"5-690" },
     { "5-7000",		"5-8396" },
     { "5-85001",		"5-89996" },
     { "5-900165",		"5-900850" },

     /* No 6- ISBN assignments */

     /* China */
     { "7-00",			"7-04" },
     { "7-100",			"7-314" },
     { "7-5000",		"7-5639" },
     { "7-80000",		"7-81035" },
     { "7-900000",		"7-900000" },

     /* Czechoslovakia (Czech Republic, Slovakia) */
     { "80-00",			"80-09" },
     { "80-200",		"80-236" },
     { "80-7000",		"80-7151" },
     { "80-85000",		"80-85529" },
     { "80-900075",		"80-901081" },

     /* India */
     { "81-200",		"81-648" },
     { "81-7000",		"81-7266" },
     { "81-85000",		"81-85690" },
     { "81-900000",		"81-900211" },

     /* Norway */
     { "82-400",		"82-599" },
     { "82-7000",		"82-7661" },
     { "82-90000",		"82-91769" },
     { "82-990000",		"82-992517" },

     /* Poland */
     { "83-00",			"83-11" },
     { "83-200",		"83-233" },
     { "83-7000",		"83-7080" },
     { "83-85000",		"83-85541" },
     { "83-900000",		"83-900482" },

     /* Spain */
     { "84-00",			"84-07" },
     { "84-200",		"84-699" },
     { "84-7000",		"84-8420" },
     { "84-85000",		"84-89600" },

     /* Brazil */
     { "85-00",			"85-17" },
     { "85-200",		"85-341" },
     { "85-7000",		"85-7261" },
     { "85-85000",		"85-85457" },
     { "85-900000",		"85-900051" },

     /* Bosnia and Herzegovina, Croatia, Macedonia, Slovenia, Yugoslavia */
     { "86-03",			"86-27" },
     { "86-301",		"86-455" },
     { "86-7001",		"86-7861" },
     { "86-80001",		"86-82045" },
     { "86-900941",		"86-901289" },

     /* Denmark */
     { "87-00",			"87-20" },
     { "87-411",		"87-611" },
     { "87-7000",		"87-7799" },
     { "87-85001",		"87-89796" },
     { "87-980000",		"87-997202" },

     /* Italy, Switzerland (Italian) */
     { "88-00",			"88-19" },
     { "88-200",		"88-461" },
     { "88-7000",		"88-8402" },
     { "88-81785",		"88-86065" },

     /* Republic of Korea */
     { "89-0",			"89-0" /* <--no assignments yet */ },

     /* Belgium (Flemish), Netherlands */
     { "90-00",			"90-18" },
     { "90-200",		"90-449" },
     { "90-5000",		"90-6999" },
     { "90-70000",		"90-74319" },
     { "90-800000",		"90-800868" },

     /* Sweden */
     { "91-0",			"91-1" },
     { "91-20",			"91-48" },
     { "91-500",		"91-632" },
     { "91-7000",		"91-7998" },
     { "91-85002",		"91-88356" },
     { "91-970000",		"91-992083" },

     /* International Publishers (UNESCO) */
     { "92-0",			"92-5" },
     { "92-61",			"92-77" },
     { "92-800",		"92-891" },
     { "92-9001",		"92-9117" },

     /* India */
     { "93-0",			"93-0" /* <--no assignments yet */ },

     /* Argentina */
     { "950-00",		"950-47" },
     { "950-500",		"950-795" },
     { "950-9000",		"950-9899" },
     { "950-99000",		"950-99949" },

     /* Finland */
     { "951-0",			"951-1" },
     { "951-20",		"951-54" },
     { "951-550",		"951-889" },
     { "951-8900",		"951-9498" },
     { "951-95000",		"951-96448" },

     /* Finland */
     { "952-90",		"952-90" },
     { "952-6666",		"952-6666" },
     { "952-9500",		"952-9714" },

     /* Croatia (publisher range unknown) */
     { "953-00",		"953-99" },

     /* Bulgaria */
     { "954-0",			"954-0" /* <--no assignments yet */ },

     /* Sri Lanka */
     { "955-20",		"955-28" },
     { "955-550",		"955-616" },
     { "955-9000",		"955-9151" },
     { "955-95000",		"955-95444" },

     /* Chile */
     { "956-10",		"956-19" },
     { "956-200",		"956-266" },
     { "956-7000",		"956-7205" },

     /* Taiwan (Republic of China) */
     { "957-05",		"957-43" },
     { "957-500",		"957-685" },
     { "957-8500",		"957-9699" },

     /* Colombia */
     { "958-02",		"958-32" },
     { "958-600",		"958-656" },
     { "958-9000",		"958-9285" },
     { "958-95001",		"958-95343" },

     /* Cuba */
     { "959-00",		"959-13" },
     { "959-200",		"959-216" },
     { "959-7000",		"959-7033" },

     /* Greece */
     { "960-00",		"960-12" },
     { "960-200",		"960-433" },
     { "960-7000",		"960-8499" },
     { "960-85000",		"960-85203" },

     /* Slovenia (publisher range unknown) */
     { "961-00",		"961-99" },

     /* Hong Kong */
     { "962-00",		"962-19" },
     { "962-201",		"962-474" },
     { "962-7001",		"962-7646" },

     /* Hungary */
     { "963-00",		"963-18" },
     { "963-200",		"963-892" },
     { "963-700",		"963-8481" },
     { "963-85000",		"963-85084" },

     /* Iran (publisher range unknown) */
     { "964-00",		"964-99" },

     /* Israel */
     { "965-01",		"965-19" },
     { "965-207",		"965-442" },

     /* Ukraine  (publisher range unknown) */
     { "966-00",		"966-99" },

     /* Malaysia */
     { "967-60",		"967-89" },
     { "967-900",		"967-989" },
     { "967-9900",		"967-9989" },
     { "967-99901",		"967-99999" },

     /* Mexico */
     { "968-10",		"968-39" },
     { "968-400",		"968-899" },
     { "968-6000",		"968-7275" },

     /* Pakistan */
     { "969-0",			"969-1" },
     { "969-26",		"969-39" },
     { "969-400",		"969-473" },
     { "969-8000",		"969-8159" },

     /* Mexico */
     { "970-05",		"970-10" },
     { "970-604",		"970-619" },
     { "970-91000",		"970-91074" },

     /* Philippines */
     { "971-06",		"971-36" },
     { "971-500",		"971-631" },
     { "971-8500",		"971-8819" },
     { "971-91000",		"971-91273" },

     /* Portugal */
     { "972-0",			"972-1" },
     { "972-20",		"972-50" },
     { "972-550",		"972-722" },
     { "972-8004",		"972-9499" },
     { "972-95000",		"972-97520" },

     /* Romania */
     { "973-21",		"973-49" },
     { "973-550",		"973-682" },
     { "973-9000",		"973-9134" },
     { "973-95000",		"973-95521" },

     /* Thailand */
     { "974-00",		"974-10" },
     { "974-200",		"974-685" },
     { "974-7000",		"974-8499" },
     { "974-85000",		"974-88000" },

     /* Turkey */
     { "975-09",		"975-19" },
     { "975-345",		"975-554" },
     { "975-7402",		"975-7797" },
     { "975-95384",		"975-96606" },

     /* Caribbean Community (CARICOM): Antigua, Bahamas, Barbados, */
     /* Belize, Dominica, Grenada, Guyana, Jamaica, Montserrat, Saint */
     /* Kitts and Nevis, Saint Lucia, Saint Vincent, Trinidad and Tobago */
     { "976-40",		"976-42" },
     { "976-600",		"976-636" },
     { "976-8000",		"976-8105" },

     /* Egypt */
     { "977-01",		"977-17" },
     { "977-200",		"977-457" },
     { "977-5000",		"977-5235" },

     /* Nigeria */
     { "978-000",		"978-199" },
     { "978-2000",		"978-2887" },
     { "978-30000",		"978-31118" },

     /* Indonesia */
     { "979-400",		"979-557" },
     { "979-8000",		"979-8322" },

     /* Venezuela */
     { "980-00",		"980-07" },
     { "980-200",		"980-316" },
     { "980-6001",		"980-6303" },

     /* Singapore */
     { "981-00",		"981-03" },
     { "981-200",		"981-215" },
     { "981-3000",		"981-3099" },

     /* South Pacific, Cook Islands, Fiji, Kiribati, Nauru, Niue, */
     /* Solomon Islands, Tokelau, Tonga, Tuvalu, Vanuatu, Samoa */
     { "982-01",		"982-03" },
     { "982-100",		"982-500" },

     /* Malaysia */
     { "983-60",		"983-73" },
     { "983-800",		"983-899" },
     { "983-9000",		"983-9750" },
     { "983-99382",		"983-99747" },

     /* Bangladesh */
     { "984-01",		"984-30" },
     { "984-400",		"984-556" },
     { "984-8005",		"984-8085" },

     /* Belarus (publisher range unknown) */
     { "985-00",		"985-99" },

     /* Argentina */
     { "987-0",			"987-0" /* <--no assignments yet */ },

     /* Libya (publisher range unknown) */
     { "9959-0",		"9959-9" },

     /* Algeria (publisher range unknown) */
     { "9961-0",		"9961-9" },

     /* Panama (publisher range unknown) */
     { "9962-0",		"9962-9" },

     /* Cyprus */
     { "9963-0",		"9963-1" },
     { "9963-30",		"9963-44" },
     { "9963-550",		"9963-599" },
     { "9963-7500",		"9963-7846" },

     /* Ghana */
     { "9964-0",		"9964-6" },
     { "9964-70",		"9964-94" },
     { "9964-950",		"9964-990" },

     /* Kazakhstan (publisher range unknown) */
     { "9965-0",		"9965-9" },

     /* Kenya */
     { "9966-20",		"9966-49" },
     { "9966-830",		"9966-882" },
     { "9966-9840",		"9966-9874" },

     /* Kyrgyzstan (publisher range unknown) */
     { "9967-0",		"9967-9" },

     /* Costa Rica */
     { "9968-9702",		"9968-9703" },

     /* Uganda (publisher range unknown) */
     { "9970-0",		"9970-9" },

     /* Singapore */
     { "9971-0",		"9971-4" },
     { "9971-60",		"9971-89" },
     { "9971-900",		"9971-989" },
     { "9971-9900",		"9971-9924" },

     /* Peru (publisher range unknown) */
     { "9972-0",		"9972-9" },

     /* Tunisia */
     { "9973-10",		"9973-18" },
     { "9973-700",		"9973-925" },
     { "9973-9700",		"9973-9918" },

     /* Uruguay */
     { "9974-0",		"9974-0" },
     { "9974-30",		"9974-36" },
     { "9974-550",		"9974-580" },
     { "9974-7500",		"9974-7505" },

     /* Moldova (publisher range unknown) */
     { "9975-0",		"9975-9" },

     /* Tanzania */
     { "9976-1",		"9976-5" },
     { "9976-60",		"9976-84" },
     { "9976-900",		"9976-988" },
     { "9976-9990",		"9976-9992" },

     /* Costa Rica */
     { "9977-00",		"9977-89" },
     { "9977-900",		"9977-989" },
     { "9977-9900",		"9977-9999" },

     /* Ecuador */
     { "9978-57",		"9978-99" },
     { "9978-951",		"9978-971" },
     { "9978-9904",		"9978-9910" },

     /* Iceland */
     { "9979-1",		"9979-4" },
     { "9979-50",		"9979-57" },
     { "9979-800",		"9979-827" },
     { "9979-9000",		"9979-9067" },

     /* Papua New Guinea */
     { "9980-0",		"9980-1" },
     { "9980-54",		"9980-84" },
     { "9980-900",		"9980-915" },

     /* Morocco */
     { "9981-0",		"9981-0" /* <-- no assignments yet */ },

     /* Zambia */
     { "9982-00",		"9982-30" },
     { "9982-800",		"9982-815" },
     { "9982-9900",		"9982-9900" },

     /* Gambia */
     { "9983-85",		"9983-87" },
     { "9983-9900",		"9983-9901" },

     /* Latvia (publisher range unknown) */
     { "9984-0",		"9984-9" },

     /* Estonia (publisher range unknown) */
     { "9985-0",		"9985-9" },

     /* Lithuania (publisher range unknown) */
     { "9986-0",		"9986-9" },

     /* Tanzania */
     { "9987-30",		"9987-30" },
     { "9987-550",		"9987-583" },
     { "9987-8800",		"9987-8823" },

     /* Ghana */
     { "9988-0",		"9988-0" /* <--no assignments yet */ },

     /* Macedonia (publisher range unknown) */
     { "9989-0",		"9989-9" },

     /* Mauritius */
     { "99903-0",		"99903-0" /* <--no assignments yet */ },

     /* Netherlands Antilles */
     { "99904-0",		"99904-4" },
     { "99904-60",		"99904-65" },
     { "99904-900",		"99904-917" },

     /* Malawi */
     { "99908-11",		"99908-29" },
     { "99908-900",		"99908-900" },

     /* Malta (publisher range unknown) */
     { "99909-0",		"99909-9" },

     /* Lesotho */
     { "99911-00",		"99911-32" },
     { "99911-600",		"99911-860" },

     /* Botswana */
     { "99912-0",		"99912-5" },
     { "99912-60",		"99912-79" },
     { "99912-900",		"99912-900" },

     /* Andorra */
     { "99913-0",		"99913-0" /* <--no assignments yet */ },

     /* Suriname */
     { "99914-0",		"99914-4" },
     { "99914-50",		"99914-58" },
     { "99914-900",		"99914-927" },

     /* Maldives */
     { "99915-0",		"99915-3" },
     { "99915-50",		"99915-78" },
     { "99915-800",		"99915-830" },

     /* Namibia */
     { "99916-1",		"99916-1" },
     { "99916-30",		"99916-35" },
     { "99916-700",		"99916-704" },

     /* Benin (publisher range unknown) */
     { "99919-0",		"99919-9" },

     /* Andorra */
     { "99920-0",		"99920-2" },
     { "99920-50",		"99920-55" },
     { "99920-900",		"99920-904" },

     /* Qatar (publisher range unknown) */
     { "99921-0",		"99921-9" },

     /* Guatemala (publisher range unknown) */
     { "99922-0",		"99922-9" },

     /* Nicaragua (publisher range unknown) */
     { "99924-0",		"99924-9" },

     /* From http://www.bowker.com/standards/home/isbn/international/hyphenation-instructions.html
	The publisher prefix ranges in the English group (U.S., U.K.,
	Canada, Australia, New Zealand, etc) are as follows:

	     Group                   If Number Ranges         Insert Hyphens
	     Identifier "0"           are Between              After
	     -------------------------------------------------------------------
	     00.......19                00-19             1st  3rd  9th digit
	     200......699               20-69              "   4th       "
	     7000.....8499              70-84              "   5th       "
	     85000....89999             85-89              "   6th       "
	     900000...949999            90-94              "   7th       "
	     9500000..9999999           95-99              "   8th       "



	     Group                   If Number Ranges        Insert Hyphens
	     Identifier "1"           are Between             After
	     -------------------------------------------------------------------
	     55000....86979           5500-8697           1st  6th  9th digit
	     869800...998999          8698-9989            "   7th       "
	     9990000..9999999         9990-9999            "   8th       "



	       The following table gives the range distribution of the group identifiers:

					0 - 7
				       80 - 94
				      950 - 995
				     9960 - 9989
				    99900 - 99999

	These ranges are already partially covered by ranges above, but
	we include them here for completeness, unless they are identical
	to one above, in which case, they are commented out.  I have yet
	to find comparable data for other language groups. */


     /* { "0-00",		"0-19" }, */
     { "0-200",			"0-699" },
     /* { "0-7000",		"0-8499" }, */
     { "0-85000",		"0-89999" },
     /* { "0-900000",		"0-949999" }, */
     { "0-9500000",		"0-9999999" },

     { "1-55000",		"1-86979" },
     { "1-869800",		"1-998999" },
     { "1-9990000",		"1-9999999" },

     { (const char*)NULL,	(const char*)NULL }
    };

    for (i = 0; (ISBN_range[i].begin != (const char*)NULL); ++i)
    {
	if (in_ISBN_range(ISBN_range[i].begin, ISBN, ISBN_range[i].end)
	    == 0)
	    return (hyphenate_one_ISBN(ISBN_range[i].begin, ISBN));
    }
    return ((const char*)NULL);
}

#define skip_non_ISBN_digit(p)    while (*p && !isISBNdigit(*p)) p++

#if defined(HAVE_STDC)
static const char *
hyphenate_one_ISBN(const char *prefix, const char *ISBN)
#else /* K&R style */
static const char *
hyphenate_one_ISBN(prefix,ISBN)
const char *prefix;
const char *ISBN;
#endif
{
    /*******************************************************************
      Given a countrygroupnumber-publishernumber prefix, and an ISBN
      optionally containing spaces and hyphens, return a pointer to an
      unmodifiable properly-hyphenated ISBN stored in an internal buffer
      that is overwritten on subsequent calls, or NULL if the correct
      number of ISBN digits is not found.

      The input ISBN can contain optional leading and trailing text,
      such as a line from a BibTeX .bib file, like this:

	 ISBN =         "0-387-09823-2 (paperback)",

     ******************************************************************/

    static char new_ISBN[MAX_ISBN];
    int k;

    skip_non_ISBN_digit(ISBN);

    for (k = 0; *ISBN && (k < (MAX_ISBN - 2)); )
    {
	if (*prefix == '-')
	{
	    new_ISBN[k++] = '-';
	    prefix++;
	}
	else if (*prefix)
	{
	    skip_non_ISBN_digit(ISBN);
	    if (*ISBN == '\0')
		break;
	    new_ISBN[k++] = *ISBN++;
	    prefix++;
	    if ((*prefix == '\0') && (k < MAX_ISBN))
		new_ISBN[k++] = '-';
	}
	else			/* past prefix */
	{
	    skip_non_ISBN_digit(ISBN);
	    if (*ISBN == '\0')
		break;
	    new_ISBN[k++] = *ISBN++;
	}
    }
    if ((k == (MAX_ISBN - 2)) && !isISBNdigit(*ISBN))
    {
	new_ISBN[(MAX_ISBN - 2)] = new_ISBN[(MAX_ISBN - 3)];
				/* move checksum digit to end */
	new_ISBN[(MAX_ISBN - 3)] = '-';	/* prefix it with a hyphen */
	new_ISBN[(MAX_ISBN - 1)] = '\0'; /* terminate the string */
	return ((const char*)&new_ISBN[0]);
    }
    else
	return ((const char*)NULL);
}

#if defined(HAVE_STDC)
void
ISBN_hyphenate(char *s,char *t,size_t maxs)
#else /* K&R style */
void
ISBN_hyphenate(s,t,maxs)
char *s;
char *t;
size_t maxs;
#endif
{
    const char *p;
    const char *r;
    const char *next;
    const char *start;

    /* Given a string s[] containing one or more ISBNs, rewrite the */
    /* string in-place with correct ISBN hyphenation.  Up to maxs-1 */
    /* non-NUL characters of s[] may be used. t[] is workspace, at */
    /* least as large as s[].   If insufficient workspace is */
    /* available, s[] is returned unchanged. */

    t[0] = '\0';

#if defined(__WATCOMC__)
    /* Watcom 10.0 C++ compilers on IBM PC cannot handle the original
       version, which was written that way to avoid compiler warnings,
       sigh... */
    for (p = start = s; (p = next_ISBN(p,&next)) != (const char*)NULL;
	 start = p)
#else
    for (p = start = s; (p = next_ISBN(p,&next), p) != (const char*)NULL;
	 start = p)
#endif
    {
	if ((strlen(t) + (size_t)(p-start)) >= maxs)
	    return;		/* insufficient space: premature return */
	(void)strncat(t,start,(size_t)(p-start));
	r = fix_ISBN(p);
	if (r != (char*)NULL)
	{
	    if ((strlen(t) + strlen(r)) >= maxs)
		return;		/* insufficient space: premature return */
	    (void)strcat(t,r);
	    p = next;
	}
	else
	{
	    if ((strlen(t) + 1) >= maxs)
		return;		/* insufficient space: premature return */
	    (void)strncat(t,p,1);
	    ++p;
	}
    }
    if ((strlen(t) + strlen(start)) >= maxs)
	return;		/* insufficient space: premature return */
    (void)strcat(t,start);
    (void)strcpy(s,t);
}

#if defined(HAVE_STDC)
static int
in_ISBN_range(const char *begin, const char *ISBN, const char *end)
#else /* K&R style */
static int
in_ISBN_range(begin,ISBN,end)
const char *begin;
const char *ISBN;
const char *end;
#endif
{
    /* Compare the countrygroupnumber-publishernumber part of ISBN
       against the range (begin, end), and return -1 (less than),
       0 (in range), or +1 (greater than). */

    char begin_prefix[MAX_ISBN];
    char end_prefix[MAX_ISBN];
    char ISBN_prefix[MAX_ISBN];

    squeeze_ISBN(begin_prefix, begin);
    squeeze_ISBN(ISBN_prefix,ISBN);

    if (strncmp(ISBN_prefix,begin_prefix,strlen(begin_prefix)) < 0)
	return (-1);

    squeeze_ISBN(end_prefix,end);
    if (strncmp(end_prefix,ISBN_prefix,strlen(end_prefix)) < 0)
	return (1);

    return (0);
}

#if defined(HAVE_STDC)
static void
squeeze_ISBN(char * out_ISBN, const char *in_ISBN)
#else /* K&R style */
static void
squeeze_ISBN(out_ISBN,in_ISBN)
char * out_ISBN;
const char *in_ISBN;
#endif
{		/* Copy in_ISBN to out_ISBN, eliminating non-ISBN characters */
    char *limit = out_ISBN + MAX_ISBN;

    for ( ; out_ISBN < limit ; )
    {
	skip_non_ISBN_digit(in_ISBN);
	*out_ISBN = *in_ISBN;
	if (*in_ISBN == '\0')
	    break;
	in_ISBN++;
	out_ISBN++;
    }
}
