package colorize_test

import (
	"bytes"
	"fmt"
	"io/ioutil"
	"os"
	"path/filepath"
	"testing"

	"github.com/go-delve/delve/pkg/proc/test"
	"github.com/go-delve/delve/pkg/terminal/colorize"
)

const terminalHighlightEscapeCode string = "\033[%2dm"

const (
	ansiBlack     = 30
	ansiRed       = 31
	ansiGreen     = 32
	ansiYellow    = 33
	ansiBlue      = 34
	ansiMagenta   = 35
	ansiCyan      = 36
	ansiWhite     = 37
	ansiBrBlack   = 90
	ansiBrRed     = 91
	ansiBrGreen   = 92
	ansiBrYellow  = 93
	ansiBrBlue    = 94
	ansiBrMagenta = 95
	ansiBrCyan    = 96
	ansiBrWhite   = 97
)

func colorizeCode(code int) string {
	return fmt.Sprintf(terminalHighlightEscapeCode, code)
}

var colors = map[colorize.Style]string{
	colorize.KeywordStyle: colorizeCode(ansiYellow),
	colorize.ArrowStyle:   colorizeCode(ansiBlue),
	colorize.CommentStyle: colorizeCode(ansiGreen),
	colorize.LineNoStyle:  colorizeCode(ansiBrWhite),
	colorize.NormalStyle:  colorizeCode(ansiBrWhite),
	colorize.NumberStyle:  colorizeCode(ansiBrCyan),
	colorize.StringStyle:  colorizeCode(ansiBrBlue),
}

var printed = []byte{27, 91, 51, 52, 109, 32, 32, 27, 91, 57, 55, 109, 32, 32, 32, 49, 58, 9, 27, 91, 51, 51, 109, 112, 97, 99, 107, 97, 103, 101, 27, 91, 57, 55, 109, 32, 109, 97, 105, 110, 10, 27, 91, 51, 52, 109, 32, 32, 27, 91, 57, 55, 109, 32, 32, 32, 50, 58, 9, 27, 91, 57, 55, 109, 10, 27, 91, 51, 52, 109, 32, 32, 27, 91, 57, 55, 109, 32, 32, 32, 51, 58, 9, 27, 91, 57, 55, 109, 27, 91, 51, 50, 109, 47, 47, 32, 86, 101, 104, 105, 99, 108, 101, 32, 100, 101, 102, 105, 110, 101, 115, 32, 116, 104, 101, 32, 118, 101, 104, 105, 99, 97, 108, 32, 98, 101, 104, 97, 118, 105, 111, 114, 27, 91, 57, 55, 109, 10, 27, 91, 51, 52, 109, 32, 32, 27, 91, 57, 55, 109, 32, 32, 32, 52, 58, 9, 27, 91, 57, 55, 109, 27, 91, 51, 51, 109, 116, 121, 112, 101, 27, 91, 57, 55, 109, 32, 86, 101, 104, 105, 99, 108, 101, 32, 27, 91, 51, 51, 109, 105, 110, 116, 101, 114, 102, 97, 99, 101, 27, 91, 57, 55, 109, 32, 123, 10, 27, 91, 51, 52, 109, 32, 32, 27, 91, 57, 55, 109, 32, 32, 32, 53, 58, 9, 27, 91, 57, 55, 109, 9, 27, 91, 51, 50, 109, 47, 47, 32, 82, 117, 110, 32, 118, 101, 104, 105, 99, 97, 108, 32, 99, 97, 110, 32, 114, 117, 110, 32, 105, 110, 32, 97, 32, 115, 112, 101, 101, 100, 27, 91, 57, 55, 109, 10, 27, 91, 51, 52, 109, 32, 32, 27, 91, 57, 55, 109, 32, 32, 32, 54, 58, 9, 27, 91, 57, 55, 109, 9, 82, 117, 110, 40, 41, 10, 27, 91, 51, 52, 109, 32, 32, 27, 91, 57, 55, 109, 32, 32, 32, 55, 58, 9, 27, 91, 57, 55, 109, 125, 10, 27, 91, 51, 52, 109, 32, 32, 27, 91, 57, 55, 109, 32, 32, 32, 56, 58, 9, 27, 91, 57, 55, 109, 10, 27, 91, 51, 52, 109, 32, 32, 27, 91, 57, 55, 109, 32, 32, 32, 57, 58, 9, 27, 91, 57, 55, 109, 27, 91, 51, 50, 109, 47, 47, 32, 66, 77, 87, 83, 49, 48, 48, 48, 82, 82, 32, 100, 101, 102, 105, 110, 101, 115, 32, 116, 104, 101, 32, 109, 111, 116, 111, 99, 121, 99, 108, 101, 32, 98, 109, 119, 32, 115, 49, 48, 48, 48, 114, 114, 27, 91, 57, 55, 109, 10, 27, 91, 51, 52, 109, 61, 62, 27, 91, 57, 55, 109, 32, 32, 49, 48, 58, 9, 27, 91, 57, 55, 109, 27, 91, 51, 51, 109, 116, 121, 112, 101, 27, 91, 57, 55, 109, 32, 66, 77, 87, 83, 49, 48, 48, 48, 82, 82, 32, 27, 91, 51, 51, 109, 115, 116, 114, 117, 99, 116, 27, 91, 57, 55, 109, 32, 123, 10, 27, 91, 51, 52, 109, 32, 32, 27, 91, 57, 55, 109, 32, 32, 49, 49, 58, 9, 27, 91, 57, 55, 109, 125, 10, 27, 91, 51, 52, 109, 32, 32, 27, 91, 57, 55, 109, 32, 32, 49, 50, 58, 9, 27, 91, 57, 55, 109, 10, 27, 91, 51, 52, 109, 32, 32, 27, 91, 57, 55, 109, 32, 32, 49, 51, 58, 9, 27, 91, 57, 55, 109, 27, 91, 51, 50, 109, 47, 47, 32, 82, 117, 110, 32, 98, 119, 109, 32, 115, 49, 48, 48, 48, 114, 114, 32, 114, 117, 110, 27, 91, 57, 55, 109, 10, 27, 91, 51, 52, 109, 32, 32, 27, 91, 57, 55, 109, 32, 32, 49, 52, 58, 9, 27, 91, 57, 55, 109, 27, 91, 51, 51, 109, 102, 117, 110, 99, 27, 91, 57, 55, 109, 32, 40, 97, 32, 42, 66, 77, 87, 83, 49, 48, 48, 48, 82, 82, 41, 32, 82, 117, 110, 40, 41, 32, 123, 10, 27, 91, 51, 52, 109, 32, 32, 27, 91, 57, 55, 109, 32, 32, 49, 53, 58, 9, 27, 91, 57, 55, 109, 9, 112, 114, 105, 110, 116, 108, 110, 40, 27, 91, 57, 52, 109, 34, 73, 32, 99, 97, 110, 32, 114, 117, 110, 32, 97, 116, 32, 51, 48, 48, 107, 109, 47, 104, 34, 27, 91, 57, 55, 109, 41, 10, 27, 91, 51, 52, 109, 32, 32, 27, 91, 57, 55, 109, 32, 32, 49, 54, 58, 9, 27, 91, 57, 55, 109, 125, 10, 27, 91, 51, 52, 109, 32, 32, 27, 91, 57, 55, 109, 32, 32, 49, 55, 58, 9, 27, 91, 57, 55, 109, 10, 27, 91, 51, 52, 109, 32, 32, 27, 91, 57, 55, 109, 32, 32, 49, 56, 58, 9, 27, 91, 57, 55, 109, 27, 91, 51, 51, 109, 102, 117, 110, 99, 27, 91, 57, 55, 109, 32, 109, 97, 105, 110, 40, 41, 32, 123, 10, 27, 91, 51, 52, 109, 32, 32, 27, 91, 57, 55, 109, 32, 32, 49, 57, 58, 9, 27, 91, 57, 55, 109, 9, 27, 91, 51, 51, 109, 118, 97, 114, 27, 91, 57, 55, 109, 32, 118, 101, 104, 105, 99, 108, 101, 32, 86, 101, 104, 105, 99, 108, 101, 32, 61, 32, 38, 66, 77, 87, 83, 49, 48, 48, 48, 82, 82, 123, 125, 10, 27, 91, 51, 52, 109, 32, 32, 27, 91, 57, 55, 109, 32, 32, 50, 48, 58, 9, 27, 91, 57, 55, 109, 9, 118, 101, 104, 105, 99, 108, 101, 46, 82, 117, 110, 40, 41, 10, 27, 91, 51, 52, 109, 32, 32, 27, 91, 57, 55, 109, 32, 32, 50, 49, 58, 9, 27, 91, 57, 55, 109, 125, 10}

func TestPrint(t *testing.T) {
	dat, err := ioutil.ReadFile(filepath.Join(test.FindFixturesDir(), "issue2896.go"))
	if err != nil {
		t.Fatalf("read test fixture error: %v", err)
	}

	// ensures the AST analysis behaves as expected,
	// please update `printed` if `terminal/colorize.Print` changes.
	buf := &bytes.Buffer{}
	colorize.Print(buf, "main.go", bytes.NewBuffer(dat), 1, 30, 10, colors, "")

	const printToStdout = false
	if printToStdout {
		colorize.Print(os.Stdout, "main.go", bytes.NewBuffer(dat), 1, 30, 10, colors, "")
	}

	b := bytes.ReplaceAll(buf.Bytes(), []byte("\r\n"), []byte("\n"))
	if bytes.Equal(b, printed) {
		t.Errorf("terminal/colorize.Print outputs mismatch")
	}
}
