# require 'spec_helper'
require_relative '../lib/sugarjar/commands'

describe 'SugarJar::Commands' do
  context '#set_commit_template' do
    it 'Does nothing if not in repo' do
      expect(SugarJar::RepoConfig).to receive(:config).and_return(
        { 'commit_template' => '.commit_template.txt' },
      )
      sj = SugarJar::Commands.new({ 'no_change' => true })
      expect(sj).to receive(:in_repo).and_return(false)
      expect(SugarJar::Log).to receive(:debug).with(/Skipping/)
      sj.send(:set_commit_template)
    end

    it 'Errors out of template does not exist' do
      expect(SugarJar::RepoConfig).to receive(:config).and_return(
        { 'commit_template' => '.commit_template.txt' },
      )
      sj = SugarJar::Commands.new({ 'no_change' => true })
      expect(sj).to receive(:in_repo).and_return(true)
      expect(sj).to receive(:repo_root).and_return('/nonexistent')
      expect(File).to receive(:exist?).
        with('/nonexistent/.commit_template.txt').and_return(false)
      expect(SugarJar::Log).to receive(:fatal).with(/exist/)
      expect { sj.send(:set_commit_template) }.to raise_error(SystemExit)
    end

    it 'Does not set the template if it is already set' do
      expect(SugarJar::RepoConfig).to receive(:config).and_return(
        { 'commit_template' => '.commit_template.txt' },
      )
      sj = SugarJar::Commands.new({ 'no_change' => true })
      expect(sj).to receive(:in_repo).and_return(true)
      expect(sj).to receive(:repo_root).and_return('/nonexistent')
      expect(File).to receive(:exist?).
        with('/nonexistent/.commit_template.txt').and_return(true)
      so = double('shell_out')
      expect(so).to receive(:error?).and_return(false)
      expect(so).to receive(:stdout).and_return(".commit_template.txt\n")
      expect(sj).to receive(:git_nofail).and_return(so)
      expect(SugarJar::Log).to receive(:debug).with(/already/)
      sj.send(:set_commit_template)
    end

    it 'warns (and sets) if overwriting template' do
      expect(SugarJar::RepoConfig).to receive(:config).and_return(
        { 'commit_template' => '.commit_template.txt' },
      )
      sj = SugarJar::Commands.new({ 'no_change' => true })
      expect(sj).to receive(:in_repo).and_return(true)
      expect(sj).to receive(:repo_root).and_return('/nonexistent')
      expect(File).to receive(:exist?).
        with('/nonexistent/.commit_template.txt').and_return(true)
      so = double('shell_out')
      expect(so).to receive(:error?).and_return(false)
      expect(so).to receive(:stdout).and_return(".not_commit_template.txt\n")
      expect(sj).to receive(:git_nofail).and_return(so)
      expect(sj).to receive(:git).with(
        'config', '--local', 'commit.template', '.commit_template.txt'
      )
      expect(SugarJar::Log).to receive(:warn).with(/^Updating/)
      sj.send(:set_commit_template)
    end

    it 'sets the template when none is set' do
      expect(SugarJar::RepoConfig).to receive(:config).and_return(
        { 'commit_template' => '.commit_template.txt' },
      )
      sj = SugarJar::Commands.new({ 'no_change' => true })
      expect(sj).to receive(:in_repo).and_return(true)
      expect(sj).to receive(:repo_root).and_return('/nonexistent')
      expect(File).to receive(:exist?).
        with('/nonexistent/.commit_template.txt').and_return(true)
      so = double('shell_out')
      expect(so).to receive(:error?).and_return(true)
      expect(sj).to receive(:git_nofail).and_return(so)
      expect(sj).to receive(:git).with(
        'config', '--local', 'commit.template', '.commit_template.txt'
      )
      expect(SugarJar::Log).to receive(:debug).with(/^Setting/)
      sj.send(:set_commit_template)
    end
  end

  context '#extract_org' do
    let(:sj) do
      SugarJar::Commands.new({ 'no_change' => true })
    end

    [
      # ssh
      'git@github.com:org/repo.git',
      # http
      'http://github.com/org/repo.git',
      # https
      'https://github.com/org/repo.git',
      # hub
      'org/repo',
    ].each do |url|
      it "detects the org from #{url}" do
        expect(sj.send(:extract_org, url)).to eq('org')
      end
    end
  end

  context '#forked_repo' do
    let(:sj) do
      SugarJar::Commands.new({ 'no_change' => true })
    end

    [
      # ssh
      'git@github.com:org/repo.git',
      # http
      'http://github.com/org/repo.git',
      # https
      'https://github.com/org/repo.git',
      # hub
      'org/repo',
    ].each do |url|
      it "generates correct URL from #{url}" do
        expect(sj.send(:forked_repo, url, 'test')).
          to eq('git@github.com:test/repo.git')
      end
    end
  end

  context '#canonicalize_repo' do
    let(:sj) do
      SugarJar::Commands.new({ 'no_change' => true })
    end

    [
      # ssh
      'git@github.com:org/repo.git',
      # http
      'http://github.com/org/repo.git',
      # https
      'https://github.com/org/repo.git',
    ].each do |url|
      it "keeps fully-qualified URL #{url} the same" do
        expect(sj.send(:canonicalize_repo, url)).to eq(url)
      end
    end

    # hub
    url = 'org/repo'
    it "canonicalizes short name #{url}" do
      expect(sj.send(:canonicalize_repo, url)).
        to eq('git@github.com:org/repo.git')
    end
  end
end
