/*
 * @(#)MballU.c
 *
 * Copyright 1994 - 2023  David A. Bagley, bagleyd AT verizon.net
 *
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the author not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.
 *
 * This program is distributed in the hope that it will be "useful",
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */

/* Undo algorithm for Mball */

#include "MballP.h"

MballLoc  *startLoc[MAX_WEDGES] =
{
	NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL
};

static void
newStack(MballStack *s)
{
	if (s->lastMove != NULL || s->firstMove != NULL)
		return;
	if (!(s->lastMove = (MoveStack *) malloc(sizeof (MoveStack)))) {
		DISPLAY_ERROR("Not enough memory, exiting.");
	}
	if (!(s->firstMove = (MoveStack *) malloc(sizeof (MoveStack)))) {
		DISPLAY_ERROR("Not enough memory, exiting.");
	}
	s->firstMove->previous = s->lastMove->next = NULL;
	s->firstMove->next = s->lastMove;
	s->lastMove->previous = s->firstMove;
	s->count = 0;
}

static void
pushStack(MballStack *s, moveRecord **move)
{
	if (!(s->currMove = (MoveStack *) malloc(sizeof (MoveStack)))) {
		DISPLAY_ERROR("Not enough memory, exiting.");
	}
	s->lastMove->previous->next = s->currMove;
	s->currMove->previous = s->lastMove->previous;
	s->currMove->next = s->lastMove;
	s->lastMove->previous = s->currMove;
	*move = &(s->currMove->move);
	s->count++;
}

static void
popStack(MballStack *s)
{
	s->currMove = s->lastMove->previous;
	s->lastMove->previous->previous->next = s->lastMove;
	s->lastMove->previous = s->lastMove->previous->previous;
	free(s->currMove);
	s->count--;
}

static moveRecord *
topStack(MballStack *s)
{
	return (&s->lastMove->previous->move);
}

static int
emptyStack(MballStack *s)
{
	return (s->lastMove->previous == s->firstMove);
}

static void
flushStack(MballStack *s)
{
	while (s->lastMove->previous != s->firstMove) {
		s->currMove = s->lastMove->previous;
		s->lastMove->previous->previous->next = s->lastMove;
		s->lastMove->previous = s->lastMove->previous->previous;
		free(s->currMove);
	}
	s->count = 0;
}

static void
deleteStack(MballStack *s)
{
	flushStack(s);
	if (s->firstMove) {
		free(s->firstMove);
		s->firstMove = NULL;
	}
	if (s->lastMove) {
		free(s->lastMove);
		s->lastMove = NULL;
	}
}

/**********************************/

void
newMoves(MballStack *s)
{
	newStack(s);
}

void
deleteMoves(MballStack *s)
{
	deleteStack(s);
}

static void
writeMove(moveRecord *move, int direction, int control, int sector)
{
#if 0
	move->direction = direction;
	move->control = control;
#endif
	move->packed = (unsigned short int) (((direction & 0xFF) << 4) +
		(control & 0xF));
	move->sector = sector;
}

static void
readMove(moveRecord *move, int *direction, int *control, int *sector)
{
#if 0
	*direction = move->direction;
	*control = move->control;
#endif
	*control = (int) (move->packed & 0xF);
	*direction = (int) ((move->packed >> 4) & 0xFF);
	*sector = move->sector;
}

void
setMove(MballStack *s, int direction, int control, int sector)
{
	moveRecord *move;

	pushStack(s, &move);
	writeMove(move, direction, control, sector);
}

void
getMove(MballStack *s, int *direction, int *control, int *sector)
{
	readMove(topStack(s), direction, control, sector);
	popStack(s);
}

int
madeMoves(MballStack *s)
{
	return !emptyStack(s);
}

void
flushMoves(MballWidget w, MballStack *s, Boolean undo)
{
	int wedge, band;

	flushStack(s);
	if (undo) {
		for (wedge = 0; wedge < w->mball.wedges; wedge++)
			for (band = 0; band < w->mball.bands; band++) {
				startLoc[wedge][band].wedge =
					w->mball.mballLoc[wedge][band].wedge;
				startLoc[wedge][band].direction =
					w->mball.mballLoc[wedge][band].direction;
			}
	}
}

int
numMoves(MballStack *s)
{
	return s->count;
}

Boolean
scanMoves(FILE *fp, MballWidget w, int moves)
{
	int direction, control, sector, l, c;

	for (l = 0; l < moves; l++) {
		while ((c = getc(fp)) != EOF && c != SYMBOL);
		if (fscanf(fp, "%d %d %d", &direction, &control, &sector) != 3) {
			(void) fprintf(stderr,
				"corrupt scan, expecting 3 integers for move %d\n", l);
			return False;
		}
		movePuzzle(w, sector % w->mball.wedges,
			sector / w->mball.wedges, direction, control, INSTANT);
	}
	return True;
}

void
printMoves(FILE *fp, MballStack *s)
{
	int direction, sector, control, counter = 0;

	s->currMove = s->firstMove->next;
	(void) fprintf(fp, "moves\tdir\tcontrol\tsector\n");
	while (s->currMove != s->lastMove) {
		readMove(&(s->currMove->move),
			&direction, &control, &sector);
		(void) fprintf(fp, "%d%c\t%d\t%d\t%d\n",
			++counter, SYMBOL, direction, control, sector);
		s->currMove = s->currMove->next;
	}
}

Boolean
scanStartPosition(FILE *fp, MballWidget w)
{
	int wedge, band, num, c;

	while ((c = getc(fp)) != EOF && c != SYMBOL);
	for (band = 0; band < w->mball.bands; band++)
		for (wedge = 0; wedge < w->mball.wedges; wedge++) {
			if (fscanf(fp, "%d ", &num) != 1) {
				(void) fprintf(stderr,
					"corrupt start record, expecting an integer for wedge %d\n", wedge);
				return False;
			}
			startLoc[wedge][band].wedge = num;
			if (fscanf(fp, "%d ", &num) != 1) {
				(void) fprintf(stderr,
					"corrupt start record, expecting an integer for direction %d\n", wedge);
				return False;
			}
			startLoc[wedge][band].direction = num;
		}
	return True;
}

void
printStartPosition(FILE *fp, MballWidget w)
{
	int wedge, band;

	(void) fprintf(fp, "\nstartingPosition%c\n", SYMBOL);
	for (band = 0; band < w->mball.bands; band++) {
		for (wedge = 0; wedge < w->mball.wedges; wedge++) {
			(void) fprintf(fp, "%4d", startLoc[wedge][band].wedge);
			(void) fprintf(fp, "%3d",
				startLoc[wedge][band].direction);
		}
		(void) fprintf(fp, "\n");
	}
	(void) fprintf(fp, "\n");
}

void
setStartPosition(MballWidget w)
{
	int wedge, band;

	for (wedge = 0; wedge < w->mball.wedges; wedge++)
		for (band = 0; band < w->mball.bands; band++) {
			w->mball.mballLoc[wedge][band].wedge =
				startLoc[wedge][band].wedge;
			if (w->mball.orient)
				w->mball.mballLoc[wedge][band].direction =
					startLoc[wedge][band].direction;
		}
	drawAllWedges(w);
}
